import 'dart:io';

import 'package:admin_desktop/presentation/components/custom_line.dart';
import 'package:flutter/material.dart';
import 'package:flutter_esc_pos_utils/flutter_esc_pos_utils.dart';
import 'package:flutter_pos_printer_platform_image_3/flutter_pos_printer_platform_image_3.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:admin_desktop/domain/models/models.dart';
import 'package:admin_desktop/infrastructure/services/utils.dart';
import 'package:admin_desktop/presentation/components/components.dart';
import 'package:remixicon_updated/remixicon_updated.dart';
import 'package:admin_desktop/presentation/components/buttons/custom_mini_button.dart';
import 'package:admin_desktop/presentation/styles/style.dart';
import 'print_page.dart';

class GenerateCheckPage extends StatefulWidget {
  final OrderData? orderData;

  const GenerateCheckPage({super.key, required this.orderData});

  @override
  State<GenerateCheckPage> createState() => _GenerateCheckPageState();
}

class _GenerateCheckPageState extends State<GenerateCheckPage> {
  @override
  Widget build(BuildContext context) {
    return Container(
      decoration: BoxDecoration(
        color: Style.white,
        borderRadius: BorderRadius.circular(10),
      ),
      padding: REdgeInsets.symmetric(horizontal: 14, vertical: 16),
      child: SingleChildScrollView(
        child: Column(
          mainAxisSize: MainAxisSize.min,
          crossAxisAlignment: CrossAxisAlignment.center,
          children: [
            Text(AppHelpers.getTranslation(TrKeys.orderSummary),
                style: Style.interMedium(size: 20)),
            8.verticalSpace,
            Text(
                "${AppHelpers.getTranslation(TrKeys.order)} #${AppHelpers.getTranslation(TrKeys.id)}${widget.orderData?.id}",
                style: Style.interNormal(size: 16)),
            12.verticalSpace,
            CustomLine(),
            12.verticalSpace,
            Row(
              children: [
                SizedBox(
                  width: 120.w,
                  child: Text(AppHelpers.getTranslation(TrKeys.shopName),
                      style: Style.interRegular(size: 14)),
                ),
                Expanded(
                  child: Text(LocalStorage.getShop()?.translation?.title ?? "",
                      style: Style.interNormal(size: 14)),
                )
              ],
            ),
            8.verticalSpace,
            Row(
              children: [
                SizedBox(
                  width: 120.w,
                  child: Text(AppHelpers.getTranslation(TrKeys.client),
                      style: Style.interRegular(size: 14)),
                ),
                Expanded(
                  child: Text(
                      "${widget.orderData?.user?.firstname ?? ""} ${widget.orderData?.user?.lastname ?? ""}",
                      style: Style.interRegular(size: 14)),
                )
              ],
            ),
            8.verticalSpace,
            Row(
              children: [
                SizedBox(
                  width: 120.w,
                  child: Text(AppHelpers.getTranslation(TrKeys.date),
                      style: Style.interRegular(size: 14)),
                ),
                Expanded(
                  child: Text(TimeService.dateFormatMDYHm(widget.orderData?.createdAt),
                      style: Style.interRegular(size: 12)),
                )
              ],
            ),
            10.verticalSpace,
            const Divider(thickness: 2),
            SizedBox(
              height:(widget.orderData?.details?.length ?? 0)* 60.r,
              width: 1.sw/2,
              child: ListView.builder(
                  padding: const EdgeInsets.only(top: 16),
                  physics: const NeverScrollableScrollPhysics(),
                  shrinkWrap: true,
                  itemCount: widget.orderData?.details?.length ?? 0,
                  itemBuilder: (context, index) {
                    final stock = widget.orderData?.details?[index];
                    return Padding(
                      padding: const EdgeInsets.only(bottom: 16),
                      child: Column(
                        children: [
                          Row(
                            crossAxisAlignment: CrossAxisAlignment.start,
                            children: [
                              Expanded(
                                child: Text(
                                  "${stock?.stock?.product?.translation?.title ?? ""} x ${stock?.quantity ?? ""}",
                                  style: Style.interNormal(size: 14),
                                ),
                              ),
                              if (stock?.bonus ?? false)
                                Text(
                                  AppHelpers.getTranslation(TrKeys.bonus),
                                  style: Style.interNormal(size: 14),
                                ),
                              if (!(stock?.bonus ?? false))
                                Text(
                                  AppHelpers.numberFormat(
                                      number:
                                          widget.orderData?.details?[index].totalPrice),
                                  style: Style.interNormal(size: 14),
                                )
                            ],
                          ),
                          10.verticalSpace,
                         CustomLine(),
                        ],
                      ),
                    );
                  }),
            ),
          // CustomLine(),
            20.verticalSpace,
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.subtotal),
                  style: Style.interNormal(size: 14),
                ),
                const Spacer(),
                Text(
                  AppHelpers.numberFormat(number: widget.orderData?.originPrice),
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            if ((widget.orderData?.totalTax ?? 0) != 0)
              Column(
                children: [
                  10.verticalSpace,
                  Row(
                    children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.tax),
                        style: Style.interNormal(size: 14),
                      ),
                      const Spacer(),
                      Text(
                        AppHelpers.numberFormat(number: widget.orderData?.totalTax),
                        style: Style.interRegular(size: 14),
                      )
                    ],
                  ),
                ],
              ),
            if ((widget.orderData?.serviceFee ?? 0) != 0)
              Column(
                children: [
                  10.verticalSpace,
                  Row(
                    children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.serviceFee),
                        style: Style.interNormal(size: 14),
                      ),
                      const Spacer(),
                      Text(
                        AppHelpers.numberFormat(number: widget.orderData?.serviceFee),
                        style: Style.interRegular(size: 14),
                      )
                    ],
                  ),
                ],
              ),
            if ((widget.orderData?.deliveryFee ?? 0) != 0)
              Column(
                children: [
                  10.verticalSpace,
                  Row(
                    children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.deliveryFee),
                        style: Style.interNormal(size: 14),
                      ),
                      const Spacer(),
                      Text(
                        AppHelpers.numberFormat(number: widget.orderData?.deliveryFee),
                        style: Style.interRegular(size: 14),
                      )
                    ],
                  ),
                ],
              ),
            if ((widget.orderData?.totalDiscount ?? 0) != 0)
              Column(
                children: [
                  10.verticalSpace,
                  Row(
                    children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.discount),
                        style: Style.interNormal(size: 14),
                      ),
                      const Spacer(),
                      Text(
                        "-${AppHelpers.numberFormat(number: widget.orderData?.totalDiscount)}",
                        style: Style.interRegular(size: 14),
                      )
                    ],
                  ),
                ],
              ),
            if ((widget.orderData?.coupon?.price ?? 0) != 0)
              Column(
                children: [
                  10.verticalSpace,
                  Row(
                    children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.coupon),
                        style: Style.interNormal(size: 14),
                      ),
                      const Spacer(),
                      Text(
                          "-${AppHelpers.numberFormat(number: widget.orderData?.coupon?.price)}",
                          style: Style.interRegular(size: 14))
                    ],
                  ),
                ],
              ),
            10.verticalSpace,
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.totalPrice),
                  style: Style.interNormal(size: 14),
                ),
                const Spacer(),
                Text(
                  AppHelpers.numberFormat(number: widget.orderData?.totalPrice),
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            10.verticalSpace,
            Row(
              children: List.generate(
                20,
                (index) => Expanded(
                  child: Container(
                      margin: EdgeInsets.symmetric(horizontal: 4.r),
                      height: 2,
                      color: Style.iconButtonBack),
                ),
              ),
            ),
            26.verticalSpace,
            Text(
              AppHelpers.getTranslation(TrKeys.thankYou).toUpperCase(),
              style: Style.interNormal(),
            ),
            24.verticalSpace,
            Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                Expanded(
                  child: LoginButton(
                      title: AppHelpers.getTranslation(TrKeys.print),
                      onPressed: () async {
                        final bytes = await _getBytes();
                        _printEscPos(bytes,);
                      }
                  ),
                ),
                10.horizontalSpace,
                CustomMiniButton(
                  size: 56,
                  onTap: () async {
                    final bytes = await _getBytes();
                    if (context.mounted) {
                      AppHelpers.showAlertDialog(
                          context: context,
                          child: PrintPage(bytes: bytes));
                    }
                    // ignore: deprecated_member_use
                  },icon: Remix.settings_5_fill,iconColor: Style.white.withOpacity(0.7),color: Style.primary,)
              ],
            )
          ],
        ),
      ),
    );
  }

  var printerManager = PrinterManager.instance;

  List<int>? pendingTask;

  final _reconnect = false;

  final BTStatus _currentStatus = BTStatus.none;

  void _printEscPos(List<int> bytes,) async {
    final profile = await CapabilityProfile.load();
    final generator = Generator(PaperSize.mm58, profile);
    var connectedTCP = false;
    if (LocalStorage.getPrinterData() == null) return;
    var bluetoothPrinter = LocalStorage.getPrinterData()!;

    switch (bluetoothPrinter.typePrinter) {
      case PrinterType.usb:
        debugPrint("");
        bytes += generator.feed(2);
        bytes += generator.cut();
        await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: UsbPrinterInput(
                name: bluetoothPrinter.deviceName,
                productId: bluetoothPrinter.productId,
                vendorId: bluetoothPrinter.vendorId));
        pendingTask = null;
        break;
      case PrinterType.bluetooth:
        bytes += generator.cut();
        await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: BluetoothPrinterInput(
                name: bluetoothPrinter.deviceName,
                address: bluetoothPrinter.address!,
                isBle: bluetoothPrinter.isBle ?? false,
                autoConnect: _reconnect));
        pendingTask = null;
        if (Platform.isAndroid) pendingTask = bytes;
        break;
      case PrinterType.network:
        bytes += generator.feed(2);
        bytes += generator.cut();
        connectedTCP = await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: TcpPrinterInput(ipAddress: bluetoothPrinter.address!));
        if (!connectedTCP) debugPrint(' --- please review your connection ---');
        break;
    }
    if (bluetoothPrinter.typePrinter == PrinterType.bluetooth &&
        Platform.isAndroid) {
      if (_currentStatus == BTStatus.connected) {
        printerManager.send(type: bluetoothPrinter.typePrinter, bytes: bytes);
        pendingTask = null;
      }
    } else {
      printerManager.send(type: bluetoothPrinter.typePrinter, bytes: bytes);
    }
  }

  Future<List<int>> _getBytes() async {
    List<int> bytes = [];
    String customDivider = "-------------------------------------------";
    String customLine = "___________________________________________";
    String customSpace = "  ";
    final profile = await CapabilityProfile.load(name: 'XP-N160I');
    //     // PaperSize.mm80 or PaperSize.mm58
    num subTotal = 0;
    subTotal = ((widget.orderData?.details?.first.totalPrice ?? 0) -
        (widget.orderData?.totalTax ?? 0) -
        (widget.orderData?.deliveryFee ?? 0) +
        (widget.orderData?.details?.first.discount ?? 0));
    final generator = Generator(PaperSize.mm80, profile);
    bytes += generator.setGlobalCodeTable('CP1252');
    bytes += generator.setStyles(const PosStyles(align: PosAlign.center));
    bytes += generator.text(AppHelpers.getTranslation(TrKeys.orderSummary),
        styles: const PosStyles(align: PosAlign.center, bold: true));

    bytes += generator.text(
        "${AppHelpers.getTranslation(TrKeys.order)} #${AppHelpers.getTranslation(TrKeys.id)}${widget.orderData?.id}",
        styles: const PosStyles(align: PosAlign.center));
    bytes += generator.text(customDivider);
    bytes += generator.row([
      PosColumn(
          width: 7,
          text: customSpace + AppHelpers.getTranslation(TrKeys.shopName),
          styles: const PosStyles(align: PosAlign.left, codeTable: 'CP1252')),
      PosColumn(
          width: 5,
          text: LocalStorage.getShop()?.translation?.title ?? "",
          styles: const PosStyles(align: PosAlign.right, codeTable: 'CP1252')),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.client),
        styles: const PosStyles(align: PosAlign.left),
      ),
      PosColumn(
        width: 6,
        text:
            "${widget.orderData?.user?.firstname ?? ""} ${widget.orderData?.user?.lastname ?? ""}",
        styles: const PosStyles(align: PosAlign.right),
      ),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.date),
        styles: const PosStyles(align: PosAlign.left),
      ),
      PosColumn(
        width: 6,
        text: TimeService.dateFormatMDYHm(
            widget.orderData?.createdAt?.toLocal() ?? DateTime.now()),
        styles: const PosStyles(align: PosAlign.right),
      ),
    ]);

    bytes += generator.text(customLine);
    for (int index = 0; index < (widget.orderData?.details?.length ?? 0); index++) {
      bytes += generator.setStyles(const PosStyles(align: PosAlign.center));
      bytes += generator.row([
        PosColumn(
          width: 8,
          text:
              "$customSpace${widget.orderData?.details?[index].stock?.product?.translation?.title ?? ""} x ${widget.orderData?.details?[index].stock?.quantity ?? ""}",
          styles: const PosStyles(align: PosAlign.left, bold: true),
        ),
        PosColumn(
          width: 4,
          text: AppHelpers.numberFormat(
              number: widget.orderData?.details?[index].totalPrice),
          styles: const PosStyles(align: PosAlign.right, bold: true),
        ),
      ]);
    }
    bytes += generator.text(customDivider);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.subtotal),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text: AppHelpers.numberFormat(number: subTotal),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.tax),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text: AppHelpers.numberFormat(number: widget.orderData?.totalTax),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);

    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.deliveryFee),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text: AppHelpers.numberFormat(number: widget.orderData?.deliveryFee),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.discount),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text:
            AppHelpers.numberFormat(number: widget.orderData?.details?.first.discount),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.totalPrice),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text: AppHelpers.numberFormat(number: widget.orderData?.totalPrice),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);
    bytes += generator.text(customDivider);
    bytes += generator.text(AppHelpers.getTranslation(TrKeys.thankYou));
    return bytes;
  }
}
